<?php

namespace Tests\Feature;

use App\Models\Empresa;
use App\Models\EmpresaProtocolo;
use App\Models\EmpresaProtocoloPeriodo;
use App\Models\Protocolo;
use App\Models\ProtocoloResultado;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProtocoloCpfRestrictionTest extends TestCase
{
    use RefreshDatabase;

    private function setupPeriodo(string $cpfs = null): EmpresaProtocoloPeriodo
    {
        $empresa = Empresa::create([
            'name' => 'Empresa',
            'email' => 'empresa@example.com',
            'password' => bcrypt('secret'),
        ]);

        $protocolo = Protocolo::create([
            'nome' => 'Teste',
            'tipo' => 'simples',
            'status' => true,
            'permitir_pf' => true,
        ]);

        $protocolo->resultados()->create([
            'paridade' => 'geral',
            'pontuacao_de' => 0,
            'pontuacao_ate' => 100,
            'rotulo' => 'Ok',
            'texto' => 'Ok',
        ]);

        $ep = EmpresaProtocolo::create([
            'empresa_id' => $empresa->id,
            'protocolo_id' => $protocolo->id,
            'ativo' => true,
        ]);

        return EmpresaProtocoloPeriodo::create([
            'empresa_protocolo_id' => $ep->id,
            'inicio' => now()->subDay()->toDateString(),
            'fim' => now()->addDay()->toDateString(),
            'token' => 'tok',
            'cpfs' => $cpfs,
        ]);
    }

    public function test_block_when_cpf_not_allowed(): void
    {
        $periodo = $this->setupPeriodo("11111111111\n22222222222");

        $response = $this->post(route('protocolo.responder.salvar', 'tok'), [
            'cpf' => '33333333333',
            'answers' => json_encode([]),
        ]);

        $response->assertSessionHasErrors('cpf');
    }

    public function test_allow_when_cpf_in_list(): void
    {
        $periodo = $this->setupPeriodo("11111111111\n22222222222");

        $response = $this->post(route('protocolo.responder.salvar', 'tok'), [
            'cpf' => '111.111.111-11',
            'answers' => json_encode([]),
        ]);

        $response->assertStatus(200);
    }

    public function test_cpf_is_required_and_must_be_11_digits(): void
    {
        $periodo = $this->setupPeriodo();

        $response = $this->post(route('protocolo.responder.salvar', 'tok'), [
            'cpf' => '1234',
            'answers' => json_encode([]),
        ]);

        $response->assertSessionHasErrors('cpf');

        $response = $this->post(route('protocolo.responder.salvar', 'tok'), [
            'answers' => json_encode([]),
        ]);

        $response->assertSessionHasErrors('cpf');
    }
}
