<?php

namespace App\Http\Controllers;

use App\Models\EmpresaProtocoloPeriodo;
use App\Models\ProtocoloResposta;
use Illuminate\Http\Request;

class ProtocoloRespostaController extends Controller
{
    public function show(string $token)
    {
        $periodo = EmpresaProtocoloPeriodo::where('token', $token)->firstOrFail();
        $pivot = $periodo->empresaProtocolo;
        if (
            !$pivot->ativo ||
            !$pivot->protocolo->status
        ) {
            abort(404);
        }
        if (
            now()->lt($periodo->inicio->startOfDay()) ||
            now()->gt($periodo->fim->endOfDay())
        ) {
            return view('protocolos.resposta-encerrada');
        }
        $protocolo = $pivot->protocolo()->with(['perguntas.opcoes'])->first();
        $empresa = $pivot->empresa()->with('setores')->first();
        $questions = $protocolo->perguntas->map(function ($pergunta) {
            return [
                'id' => $pergunta->id,
                'title' => $pergunta->texto,
                'options' => $pergunta->opcoes->map(fn($opcao) => [
                    'value' => $opcao->pontuacao,
                    'label' => $opcao->texto,
                    'description' => '',
                ])->values(),
            ];
        })->values();

        return view('protocolos.resposta', compact('protocolo', 'empresa', 'token', 'questions'));
    }

    public function store(Request $request, string $token)
    {
        $periodo = EmpresaProtocoloPeriodo::where('token', $token)->firstOrFail();
        $pivot = $periodo->empresaProtocolo;
        if (
            !$pivot->ativo ||
            !$pivot->protocolo->status
        ) {
            abort(404);
        }
        if (
            now()->lt($periodo->inicio->startOfDay()) ||
            now()->gt($periodo->fim->endOfDay())
        ) {
            return view('protocolos.resposta-encerrada');
        }
        $answers = json_decode($request->input('answers', '[]'), true) ?? [];
        $protocolo = $pivot->protocolo()->with(['resultados', 'perguntas'])->first();
        $setorId = $request->input('setor_id');
        if ($setorId && !$pivot->empresa->setores()->where('id', $setorId)->exists()) {
            $setorId = null;
        }

        $cpf = preg_replace('/\D/', '', $request->input('cpf'));
        if (strlen($cpf) !== 11) {
            return back()->withErrors(['cpf' => 'CPF inválido.']);
        }

        $pares = 0;
        $impares = 0;
        foreach ($protocolo->perguntas as $index => $pergunta) {
            $valor = $answers[$pergunta->id] ?? 0;
            $ordem = $pergunta->ordem ?? ($index + 1);
            if ($ordem % 2 === 0) {
                $pares += $valor;
            } else {
                $impares += $valor;
            }
        }
        $total = $pares + $impares;

        if ($protocolo->tipo === 'par_impar') {
            $resultadoPar = $protocolo->resultados()
                ->where('paridade', 'par')
                ->where('pontuacao_de', '<=', $pares)
                ->where('pontuacao_ate', '>=', $pares)
                ->orderByDesc('pontuacao_de')
                ->first();
            $resultadoImpar = $protocolo->resultados()
                ->where('paridade', 'impar')
                ->where('pontuacao_de', '<=', $impares)
                ->where('pontuacao_ate', '>=', $impares)
                ->orderByDesc('pontuacao_de')
                ->first();
        } else {
            $resultado = $protocolo->resultados()
                ->where('paridade', 'geral')
                ->where('pontuacao_de', '<=', $total)
                ->where('pontuacao_ate', '>=', $total)
                ->orderByDesc('pontuacao_de')
                ->first();
        }

        $cpfList = $periodo->cpfList();
        if ($cpfList->isNotEmpty() && !$cpfList->contains($cpf)) {
            return back()->withErrors(['cpf' => 'CPF não autorizado para responder este protocolo.']);
        }
        if (
            $cpf && ProtocoloResposta::where('empresa_protocolo_periodo_id', $periodo->id)
                ->where('cpf', $cpf)
                ->exists()
        ) {
            return back()->withErrors(['cpf' => 'CPF já respondeu este formulário.']);
        }

        $data = [
            'empresa_id' => $pivot->empresa_id,
            'protocolo_id' => $protocolo->id,
            'empresa_protocolo_periodo_id' => $periodo->id,
            'nome' => $request->input('nome'),
            'cpf' => $cpf,
            'compartilhar' => $request->boolean('compartilhar'),
            'setor_id' => $setorId,
            'respostas' => $answers,
            'pontuacao_total' => $total,
        ];

        if ($protocolo->tipo === 'par_impar') {
            $data['pontuacao_par'] = $pares;
            $data['pontuacao_impar'] = $impares;
        }

        ProtocoloResposta::create($data);

        if ($protocolo->tipo === 'par_impar') {
            return view('protocolos.resposta-resultado', [
                'protocolo' => $protocolo,
                'empresa' => $pivot->empresa,
                'resultadoPar' => $resultadoPar,
                'resultadoImpar' => $resultadoImpar,
                'totalPar' => $pares,
                'totalImpar' => $impares,
            ]);
        }

        return view('protocolos.resposta-resultado', [
            'protocolo' => $protocolo,
            'empresa' => $pivot->empresa,
            'resultado' => $resultado,
            'total' => $total,
        ]);
    }

    public function checkCpf(Request $request, string $token)
    {
        $periodo = EmpresaProtocoloPeriodo::where('token', $token)->firstOrFail();
        $pivot = $periodo->empresaProtocolo;
        if (! $pivot->ativo || ! $pivot->protocolo->status) {
            abort(404);
        }
        if (now()->lt($periodo->inicio->startOfDay()) || now()->gt($periodo->fim->endOfDay())) {
            return response()->json(['allowed' => false, 'message' => 'Período encerrado'], 403);
        }
        $cpf = preg_replace('/\D/', '', $request->input('cpf'));
        if (strlen($cpf) !== 11) {
            return response()->json(['allowed' => false, 'message' => 'CPF inválido'], 422);
        }
        $cpfList = $periodo->cpfList();
        if ($cpfList->isNotEmpty() && !$cpfList->contains($cpf)) {
            return response()->json(['allowed' => false, 'message' => 'CPF não autorizado'], 403);
        }
        if ($cpf && ProtocoloResposta::where('empresa_protocolo_periodo_id', $periodo->id)->where('cpf', $cpf)->exists()) {
            return response()->json(['allowed' => false, 'message' => 'CPF já respondeu'], 409);
        }
        return response()->json(['allowed' => true]);
    }
}
