<?php

namespace App\Http\Controllers;

use App\Models\Protocolo;
use Illuminate\Http\Request;

class ProtocoloController extends Controller
{
    public function index()
    {
        $protocolos = Protocolo::all();
        return view('protocolos.index', compact('protocolos'));
    }

    public function create()
    {
        return view('protocolos.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'introducao' => 'nullable|string',
            'status' => 'required|boolean',
            'tipo' => 'required|in:simples,par_impar',
            'categoria' => 'required|in:protocolo,pesquisa',
            'visivel_empresas' => 'required|boolean',
            'permitir_pf' => 'boolean',
            'resultados' => 'array',
            'resultados.*.pontuacao_de' => 'required|integer',
            'resultados.*.pontuacao_ate' => 'required|integer',
            'resultados.*.rotulo' => 'required|string|max:255',
            'resultados.*.texto' => 'required|string',
            'resultados.*.paridade' => 'required|in:geral,par,impar',
        ]);

        $protocolo = Protocolo::create([
            'nome' => $validated['nome'],
            'introducao' => $validated['introducao'] ?? null,
            'status' => $validated['status'],
            'tipo' => $validated['tipo'],
            'categoria' => $validated['categoria'],
            'visivel_empresas' => $validated['visivel_empresas'],
            'permitir_pf' => $request->boolean('permitir_pf'),
        ]);

        foreach ($request->input('resultados', []) as $resultado) {
            $protocolo->resultados()->create($resultado);
        }

        return redirect()->route('protocolos.index')->with('success', 'Protocolo criado com sucesso.');
    }

    public function edit(Protocolo $protocolo)
    {
        $protocolo->load('resultados');
        return view('protocolos.edit', compact('protocolo'));
    }

    public function update(Request $request, Protocolo $protocolo)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'introducao' => 'nullable|string',
            'status' => 'required|boolean',
            'tipo' => 'required|in:simples,par_impar',
            'categoria' => 'required|in:protocolo,pesquisa',
            'visivel_empresas' => 'required|boolean',
            'permitir_pf' => 'boolean',
            'resultados' => 'array',
            'resultados.*.pontuacao_de' => 'required|integer',
            'resultados.*.pontuacao_ate' => 'required|integer',
            'resultados.*.rotulo' => 'required|string|max:255',
            'resultados.*.texto' => 'required|string',
            'resultados.*.paridade' => 'required|in:geral,par,impar',
        ]);

        $protocolo->update([
            'nome' => $validated['nome'],
            'introducao' => $validated['introducao'] ?? null,
            'status' => $validated['status'],
            'tipo' => $validated['tipo'],
            'categoria' => $validated['categoria'],
            'visivel_empresas' => $validated['visivel_empresas'],
            'permitir_pf' => $request->boolean('permitir_pf'),
        ]);

        $protocolo->resultados()->delete();
        foreach ($request->input('resultados', []) as $resultado) {
            $protocolo->resultados()->create($resultado);
        }

        return redirect()->route('protocolos.index')->with('success', 'Protocolo atualizado com sucesso.');
    }

    public function destroy(Protocolo $protocolo)
    {
        $protocolo->delete();
        return redirect()->route('protocolos.index')->with('success', 'Protocolo removido com sucesso.');
    }

    public function perguntas(Protocolo $protocolo)
    {
        $protocolo->load(['perguntas.opcoes']);
        return view('protocolos.perguntas', compact('protocolo'));
    }
}
