<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use App\Models\Protocolo;
use App\Models\ProtocoloResposta;
use App\Models\User;

class DashboardController extends Controller
{
    public function gestao()
    {
        $stats = [
            'totalEmpresas'    => Empresa::count(),
            'protocolosAtivos' => Protocolo::where('status', true)->count(),
            'respostasMes'     => ProtocoloResposta::whereYear('created_at', now()->year)
                ->whereMonth('created_at', now()->month)
                ->count(),
            'usuariosAtivos'   => User::count(),
        ];

        $recent = collect()
            ->merge(
                Empresa::latest()->take(5)->get()->map(fn($e) => [
                    'descricao' => 'Nova empresa cadastrada: ' . $e->name,
                    'data' => $e->created_at,
                    'color' => 'bg-green-500',
                ])
            )
            ->merge(
                Protocolo::latest()->take(5)->get()->map(fn($p) => [
                    'descricao' => 'Novo protocolo criado: ' . $p->nome,
                    'data' => $p->created_at,
                    'color' => 'bg-blue-500',
                ])
            )
            ->merge(
                ProtocoloResposta::latest()->take(5)->get()->map(fn($r) => [
                    'descricao' => 'Nova resposta registrada',
                    'data' => $r->created_at,
                    'color' => 'bg-yellow-500',
                ])
            )
            ->sortByDesc('data')
            ->take(5);

        $ranking = ProtocoloResposta::selectRaw('empresa_id, COUNT(*) as total')
            ->with('empresa:id,name')
            ->groupBy('empresa_id')
            ->orderByDesc('total')
            ->take(5)
            ->get();

        return view('dashboard-gestao', compact('stats', 'recent', 'ranking'));
    }

    public function resultados(\Illuminate\Http\Request $request)
    {
        $empresa = auth('empresa')->user();
        if (!$empresa) {
            abort(403);
        }

        $protocolos = $empresa->protocolos()->wherePivot('ativo', true)->get();
        $protocoloId = $request->query('protocolo_id');
        if ($protocoloId === null) {
            $protocoloId = $protocolos->first()?->id;
        } elseif ($protocoloId === '') {
            $protocoloId = null;
        }

        // Periodos disponíveis para o protocolo selecionado
        $periodos = collect();
        $periodoId = null;
        if ($protocoloId) {
            $pivot = \App\Models\EmpresaProtocolo::where('empresa_id', $empresa->id)
                ->where('protocolo_id', $protocoloId)
                ->first();
            if ($pivot) {
                $periodos = $pivot->periodos()->orderByDesc('inicio')->get();
                $periodoId = $request->query('periodo_id');
                if ($periodoId === null) {
                    $periodoId = $periodos->first()?->id;
                } elseif ($periodoId === '') {
                    $periodoId = null;
                }
            }
        }

        $respostasQuery = \App\Models\ProtocoloResposta::where('empresa_id', $empresa->id)
            ->where('compartilhar', true)
            ->when($protocoloId, fn($q) => $q->where('protocolo_id', $protocoloId))
            ->when($periodoId, fn($q) => $q->where('empresa_protocolo_periodo_id', $periodoId));
        $respostas = $respostasQuery->get();

        $mediaPontuacao = round($respostas->avg('pontuacao_total') ?? 0, 1);
        $totalRespostas = $respostas->count();
        $maxPontuacao = $respostas->max('pontuacao_total') ?? 0;
        $minPontuacao = $respostas->min('pontuacao_total') ?? 0;

        $monthly = $respostasQuery
            ->selectRaw('DATE_FORMAT(created_at, "%Y-%m") as mes, AVG(pontuacao_total) as media, COUNT(*) as total')
            ->groupBy('mes')
            ->orderBy('mes')
            ->get();

        $lineLabels = $monthly->map(fn($m) => \Carbon\Carbon::createFromFormat('Y-m', $m->mes)->format('M/Y'));
        $lineData = $monthly->pluck('media');
        $barLabels = $lineLabels;
        $barData = $monthly->pluck('total');

        $protocolo = $protocolos->firstWhere('id', $protocoloId);
        $categoriaLabels = [];
        $categoriaData = [];
        $categoriaPercent = [];
        if ($protocolo) {
            $protocolo->load('resultados');
            foreach ($protocolo->resultados as $cat) {
                $count = \App\Models\ProtocoloResposta::where('empresa_id', $empresa->id)
                    ->where('protocolo_id', $protocolo->id)
                    ->where('compartilhar', true)
                    ->when($periodoId, fn($q) => $q->where('empresa_protocolo_periodo_id', $periodoId))
                    ->whereBetween('pontuacao_total', [$cat->pontuacao_de, $cat->pontuacao_ate])
                    ->count();
                $categoriaLabels[] = $cat->rotulo;
                $categoriaData[] = $count;
                $categoriaPercent[] = $totalRespostas > 0 ? round(($count / $totalRespostas) * 100, 1) : 0;
            }
        }

        $setoresValidos = $empresa->setores->filter(function ($setor) use ($respostas) {
            return $respostas->where('setor_id', $setor->id)->count() >= 4;
        });

        $departamentos = [];
        if ($totalRespostas > 0) {
            foreach ($setoresValidos as $setor) {
                $count = $respostas->where('setor_id', $setor->id)->count();
                $departamentos[] = [
                    'nome' => $setor->nome,
                    'percent' => round(($count / $totalRespostas) * 100),
                ];
            }
        }

        $departamentoLabels = $lineLabels;
        $departamentoSeries = [];
        $deptMonthly = $respostasQuery
            ->selectRaw('setor_id, DATE_FORMAT(created_at, "%Y-%m") as mes, AVG(pontuacao_total) as media')
            ->groupBy('setor_id', 'mes')
            ->orderBy('mes')
            ->get();

        $palette = ['#3b82f6','#10b981','#ef4444','#f59e0b','#8b5cf6','#ec4899'];
        foreach ($setoresValidos->values() as $index => $setor) {
            $data = [];
            foreach ($monthly->pluck('mes') as $mes) {
                $record = $deptMonthly->first(fn($d) => $d->setor_id === $setor->id && $d->mes === $mes);
                $data[] = $record ? round($record->media, 1) : 0;
            }
            $departamentoSeries[] = [
                'label' => $setor->nome,
                'data' => $data,
                'borderColor' => $palette[$index % count($palette)],
                'backgroundColor' => $palette[$index % count($palette)],
                'tension' => 0.4,
            ];
        }

        return view('dashboard-resultados', compact(
            'protocolos',
            'protocoloId',
            'periodos',
            'periodoId',
            'mediaPontuacao',
            'totalRespostas',
            'maxPontuacao',
            'minPontuacao',
            'lineLabels',
            'lineData',
            'categoriaLabels',
            'categoriaData',
            'categoriaPercent',
            'barLabels',
            'barData',
            'departamentos',
            'departamentoLabels',
            'departamentoSeries'
        ));
    }
}
